/* libvcs: C function library for screen control with /dev/vcsa* devices
 * Author: Adam Ierymenko [api@one.net]
 *
 * This library is free software; it may be copied freely and you may use
 * it in your programs as long as my name remains intact.  It comes with
 * absolutely no warranties. */

/* Main virtual console access routines */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdarg.h>
#include <string.h>
#include <sys/ioctl.h>
#include <termios.h>
#include "libvcs.h"

/* Opens a console, returns NULL on error or pointer to dynamically allocated
 * vcs_handle structure */
extern struct vcs_handle *vcs_open(char *vcsname,char *ttyname)
{
	struct vcs_handle *return_handle;
	int vcs_file,realtty;
	struct winsize window_size;
	
	if ((realtty = open(ttyname,O_RDWR|O_NONBLOCK)) <= 0)
		return (struct vcs_handle *)NULL;
	if (ioctl(realtty,TIOCGWINSZ,(char *)&window_size) == -1)
		return (struct vcs_handle *)NULL;
	close(realtty);

	if ((vcs_file = open(vcsname,O_RDWR|O_NONBLOCK)) <= 0)
		return (struct vcs_handle *)NULL;
	if ((return_handle = malloc(VCS_HANDLE_SIZE)) == NULL) {
		close(vcs_file);
		return (struct vcs_handle *)NULL;
	}

	if ((return_handle->scrbuf = malloc(window_size.ws_col * window_size.ws_row * 2)) == NULL) {
		close(vcs_file);
		return (struct vcs_handle *)NULL;
	}
	return_handle->xsize = window_size.ws_col;
	return_handle->ysize = window_size.ws_row;
	return_handle->device_file = vcs_file;
	return_handle->foreground = FG_GREY;
	return_handle->background = BG_BLACK;
	strcpy(return_handle->device_filename,vcsname);
	vcs_setpos(return_handle,1,1);

	return return_handle;
}

/* Closes files and disposes of memory from a vcs_handle, and disposes of
 * the handle itself */
extern void vcs_close(struct vcs_handle *vh)
{
	close(vh->device_file);
	free(vh->scrbuf);
	free(vh);
}

/* Clears the screen, returns 0 if failed and nonzero if success */
extern int vcs_cls(struct vcs_handle *vh)
{
	register int i;
	
	if (lseek(vh->device_file,4,SEEK_SET) != 4)
		return 0;

	for (i=1;i<=(vh->xsize * vh->ysize)*2;i=i+2) {
		*(vh->scrbuf+i-1) = ' ';
		*(vh->scrbuf+i) = vh->background + vh->foreground;
	}
	
	i = (vh->xsize * vh->ysize) * 2;
	if (write(vh->device_file,vh->scrbuf,i) <= 0)
		return 0;

	vh->xpos = 1;
	vh->ypos = 1;
	lseek(vh->device_file,4,SEEK_SET);

	return 1;
}

/* Sets foreground and/or background colors */
extern void vcs_setcolors(struct vcs_handle *vh,int foreground,int background)
{
	if ((foreground + background) <= 255) {
		if (foreground >= 0)
			vh->foreground = foreground;
		if (background >= 0)
			vh->background = background;
	}
}

/* Sets current position on screen (x,y)
 * Returns 0 if failed, nonzero if successful */
extern int vcs_setpos(struct vcs_handle *vh,int xpos,int ypos)
{
	if (((xpos < 1)||(xpos > vh->xsize))||((ypos < 1)||(ypos > vh->ysize)))
		return 0;
	if (lseek(vh->device_file,(((vh->xsize * (ypos-1) * 2)+((xpos-1) * 2))+4),SEEK_SET) <= 0)
		return 0;
	vh->xpos = xpos;
	vh->ypos = ypos;

	return 1;
}

/* Prints a formatted string at current position in current color
 * Returns 0 if failed, nonzero if successful */
extern int vcs_printf(struct vcs_handle *vh,unsigned char *format, ...)
{
	va_list msg;
	unsigned char buff[16384];
	register int i;
	int c,ypos,xpos,cc,k,lc;
	
	va_start(msg,format);
	vsprintf(buff,format,msg);
	
	if (!strlen(buff))
		return 1;

	xpos = vh->xpos;
	ypos = vh->ypos;

	cc = vh->xpos - 1;
	k = 0; lc = 0;
	for (i=0;i<strlen(buff);i++) {
		switch (buff[i]) {
			case '\n':
				if ((vh->ypos + lc - 1) < vh->ysize) {
					if ((c = lseek(vh->device_file,k,SEEK_CUR)) <= 0)
						return 0;
					if (read(vh->device_file,(vh->scrbuf+k),((vh->xsize - cc) * 2)) <= 0)
						return 0;
					if (lseek(vh->device_file,(c-k),SEEK_SET) <= 0)
						return 0;
				} else bzero((vh->scrbuf+k),((vh->xsize - cc) * 2));
				k = k + ((vh->xsize - cc) * 2);
				cc = 0;
				lc++;
				break;
			case '\011':
				for (c=0;c<5;c++) {
					*(vh->scrbuf+(k++)) = ' ';
					*(vh->scrbuf+(k++)) = vh->foreground + vh->background;
					cc++;
				}
				break;
			default:
				*(vh->scrbuf+(k++)) = buff[i];
				*(vh->scrbuf+(k++)) = vh->foreground + vh->background;
				cc++;
		}
		if (cc >= vh->xsize) {
			cc = 0 + (cc - vh->xsize);
			lc++;
		}
	}
	
	if ((vh->ypos + lc - 1) >= vh->ysize) {
		memcpy(buff,vh->scrbuf,k);
		for (c=0;c<lc;c++) {
			if (!vcs_scrollup(vh))
				return 0;
			ypos--;
		}
		if (!vcs_setpos(vh,xpos,ypos))
			return 0;
		memcpy(vh->scrbuf,buff,k);
	}

	if (write(vh->device_file,vh->scrbuf,k) <= 0)
		return 0;

	vh->ypos = vh->ypos + lc;
	vh->xpos = cc + 1;

	va_end(msg);
	return 1;
}

/* Prints a single character at current position in current color
 * Returns 0 if failed, nonzero if successful */
extern int vcs_printchar(struct vcs_handle *vh,unsigned char wchar)
{
	*(vh->scrbuf+1) = vh->foreground + vh->background;
	*(vh->scrbuf) = wchar;
	if (write(vh->device_file,vh->scrbuf,2) <= 0)
		return 0;

	if ((vh->xpos+1) >= vh->xsize) {
		if (vh->ypos >= vh->ysize) {
			if (!vcs_scrollup(vh))
				return 0;
		} else vh->ypos++; 
	} else vh->xpos++;

	return 1;
}

/* Sets the color attributes of the space at the current cursor location
 * Returns 0 if failed, nonzero if successful */
extern int vcs_setattr(struct vcs_handle *vh,int foreground,int background)
{
	if ((foreground + background) > 255)
		return 0;

	if (lseek(vh->device_file,1,SEEK_CUR) <= 0)
		return 0;
	*(vh->scrbuf) = foreground + background;
	if (write(vh->device_file,vh->scrbuf,1) <= 0)
		return 0;

	return 1;
}

/* Clears from current cursor position to end of line
 * Returns 0 if failed, nonzero if successful */
extern int vcs_clreol(struct vcs_handle *vh)
{
	register int i,k = 0;
	
	for(i=0;i<=(vh->xsize - vh->xpos)+1;i++) {
		*(vh->scrbuf+(k++)) = ' ';
		*(vh->scrbuf+(k++)) = vh->foreground + vh->background;
	}
	if (write(vh->device_file,vh->scrbuf,((vh->xsize - vh->xpos)+1)*2) <= 0)
		return 0;
	if (!vcs_setpos(vh,vh->xpos,vh->ypos))
		return 0;

	return 1;
}

/* Scrolls up the screen one line and sets position to the beginning of the
 * new line.  Returns 0 if failed, nonzero if successful */
extern int vcs_scrollup(struct vcs_handle *vh)
{
	register int i;

	if (!vcs_setpos(vh,1,2))
		return 0;

	if ((i = read(vh->device_file,vh->scrbuf,((vh->xsize * (vh->ysize-1))*2))) <= 0)
		return 0;
	lseek(vh->device_file,4,SEEK_SET);
	if (write(vh->device_file,vh->scrbuf,i) <= 0)
		return 0;
	if ((!vcs_setpos(vh,1,vh->ysize))||(!vcs_clreol(vh)))
		return 0;

	return 1;
}
