/* AnsiCat: Linux ANSI picture viewer
 * Author: Adam Ierymenko [api@one.net]
 *
 * This library is free software; it may be copied freely and you may use
 * it in your programs as long as my name remains intact.  It comes with
 * absolutely no warranties. */

#include <libvcs.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>
#include <signal.h>

/* Prototypes */
void signal_handler(int signum);
static inline void do_pause(void);
static inline void do_help(char *pn);
void main(int argc,char *argv[]);
static inline void do_insecure(void);

/* Global variables */
unsigned char savedline[512];
char printing = 1,originally_root = 0;
struct vcs_handle *vh;

void signal_handler(int signum)
{
	if (signum == SIGQUIT) {
		vcs_setpos(vh,1,vh->ysize);
		write(vh->device_file,savedline,(vh->xsize * 2));
		printing = 0;
	} else exit(1);
	signal(signum,signal_handler);
}

static inline void do_pause(void)
{
	int fgcycle = FG_CYAN;
	
	usleep(700000);

	vcs_setpos(vh,1,vh->ysize);
	read(vh->device_file,savedline,(vh->xsize * 2));
	signal(SIGTERM,signal_handler);
	signal(SIGQUIT,signal_handler);

	for(;;) {
		if (vcs_setpos(vh,((vh->xsize / 2) - 25),vh->ysize)) {
			if (printing) {
				vcs_setcolors(vh,fgcycle,BG_BLACK);
				vcs_printf(vh,"Press CTRL+C to exit, CTRL+\\ to hide this message");
				if ((fgcycle++) >= FG_WHITE)
					fgcycle = FG_GREEN;
				usleep(300000);
			} else pause();
		} else exit(0);
	}
}

static inline void do_help(char *pn)
{
	printf("\
AnsiCat version 1.00 by Adam Ierymenko [api@one.net]\n\n\
Usage: %s [options] <filename>\n\
Options:\n\
   -d<delay>             - Delay for each character\n\
   -n                    - Don't pause after output\n\
   -r<delay>             - Repeat every <delay> sec (stop with CTRL+C)\n\n",pn);
	exit(1);
}

void main(int argc,char *argv[])
{
	int i,c,k,delay = 0,pause = 1,repeat = 0;
	char *filename = NULL,buff[8194],buff2[8194],*ttyname,*tmp;
	FILE *inputfile;
	
	for (i=1;i<argc;i++) {
		if (*argv[i] == '-') {
			if (!strncmp("-d",argv[i],2))
				delay = atoi(argv[i]+2);
			else if (!strcmp("-n",argv[i]))
				pause = 0;
			else if (!strncmp("-r",argv[i],2))
				repeat = atoi(argv[i]+2);
			else do_help(argv[0]);
		} else filename = argv[i];
	}
	if (filename == NULL)
		do_help(argv[0]);

	if ((inputfile = fopen(filename,"r")) == NULL) {
		printf("Could not open %s: %s\n",filename,strerror(errno));
		exit(0);
	}
	
	if ((getuid())||(geteuid()))
		setuid(0);
	else originally_root = 1;

	if ((ttyname = getenv("TTY")) == NULL) {
		printf("Environment variable TTY is not set\n");
		exit(0);
	} else {
		tmp = ttyname;
		while ((*tmp)&&(!strchr("0123456789",*tmp))) tmp++;
		if (strlen(tmp))
			i = atoi(tmp);
		sprintf(buff,"/dev/vcsa%d",i);
		tmp = &buff[256];
		if (strchr(ttyname,'/'))
			strcpy(tmp,ttyname);
		else
			sprintf(tmp,"/dev/%s",ttyname);
		if ((vh = vcs_open(buff,tmp)) == NULL) {
			sprintf(buff,"/dev/vcsa%d-",i);
			if ((vh = vcs_open(buff,tmp)) == NULL) {
				printf("Could not open %s, /dev/vcsa%d, or /dev/vcsa%d-\nRun create_vcs in the libvcs directory (as root)\nYou may also need to make this program setuid root (chmod u+s ansicat)\n");
				exit(0);
			}
		}
	}
		
do_ansi:
	vcs_setcolors(vh,FG_GREY,BG_BLACK);
	if (!vcs_cls(vh)) {
		printf("Error clearing screen\n");
		exit(0);
	}

	vcs_ansiprint(vh,NULL,0);
	while ((i = fread(buff,1,sizeof(buff),inputfile))) {
		buff[i] = '\0';
		if (strchr(buff,'\r')) {
			k = 0;
			for(c=0;c<=i;c++) {
				if (buff[c] != '\r')
					buff2[k++] = buff[c];
			}
			tmp = buff2;
		} else tmp = buff;
		vcs_ansiprint(vh,tmp,delay);
	}
	vcs_ansiprint(vh,NULL,0);
	
	if (repeat) {
		rewind(inputfile);
		sleep(repeat);
		goto do_ansi;
	}

	fclose(inputfile);
	
	if (pause)
		do_pause();

	exit(1);
}

static inline void do_insecure(void)
{
	if (!originally_root) {
		printf("Access denied\n");
		exit(0);
	}
}
