/* libvcs: C function library for screen control with /dev/vcsa* devices
 * Author: Adam Ierymenko [api@one.net]
 *
 * This library is free software; it may be copied freely and you may use
 * it in your programs as long as my name remains intact.  It comes with
 * absolutely no warranties. */

/* DOS ANSI.SYS style ansi code parser routine */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdarg.h>
#include <string.h>
#include "libvcs.h"

/* This is the value that is added to a dark color to make it a light color */
#define FG_HIGHINTENSITY FG_DARKGREY

/* Internal prototypes */
static void do_colorchange(struct vcs_handle *vh,int colorattr);
static inline char do_move_up(struct vcs_handle *vh,int count);
static inline char do_move_down(struct vcs_handle *vh,int count);
static inline char do_move_left(struct vcs_handle *vh,int count);
static inline char do_move_right(struct vcs_handle *vh,int count);
static inline char do_moveto(struct vcs_handle *vh,unsigned char *codestring);
static inline void do_set_attributes(struct vcs_handle *vh,unsigned char *codestring);

/* This sets the real fg and bg colors from an ansi attribute value */
static void do_colorchange(struct vcs_handle *vh,int colorattr)
{
	static int realfgcolor,blink,highintensity,invisible;
	
	if (colorattr == -255) {
		blink = 0;
		invisible = 0;
		realfgcolor = FG_GREY;
		highintensity = 0;
		vcs_setcolors(vh,FG_GREY,BG_BLACK);
		return;
	}
	switch (colorattr) {
		case 0:
			invisible = 0;
			highintensity = 0;
			blink = 0;
			realfgcolor = FG_GREY;
			vh->background = BG_BLACK;
			break;
		case 1:
			highintensity = FG_HIGHINTENSITY;
			break;
		case 2:
			highintensity = 0;
			break;
		case 5:
		case 6:
			if (blink)
				blink = 0;
			else
				blink = FG_BLINK;
			break;
		case 8:
			if (invisible)
				invisible = 0;
			else
				invisible++;
			break;
		case 30:
			realfgcolor = FG_BLACK;
			break;
		case 31:
			realfgcolor = FG_RED;
			break;
		case 32:
			realfgcolor = FG_GREEN;
			break;
		case 33:
			realfgcolor = FG_YELLOW;
			break;
		case 34:
			realfgcolor = FG_BLUE;
			break;
		case 35:
			realfgcolor = FG_MAGENTA;
			break;
		case 36:
			realfgcolor = FG_CYAN;
			break;
		case 37:
			realfgcolor = FG_GREY;
			break;
		case 40:
			vh->background = BG_BLACK;
			break;
		case 41:
			vh->background = BG_RED;
			break;
		case 42:
			vh->background = BG_GREEN;
			break;
		case 43:
			vh->background = BG_YELLOW;
			break;
		case 44:
			vh->background = BG_BLUE;
			break;
		case 45:
			vh->background = BG_MAGENTA;
			break;
		case 46:
			vh->background = BG_CYAN;
			break;
		case 47:
			vh->background = BG_GREY;
	}
	
	if (invisible) {
		switch (vh->background) {
			case BG_BLACK:
				vh->foreground = FG_BLACK;
				break;
			case BG_RED:
				vh->foreground = FG_RED;
				break;
			case BG_GREEN:
				vh->foreground = FG_GREEN;
				break;
			case BG_YELLOW:
				vh->foreground = FG_YELLOW;
				break;
			case BG_BLUE:
				vh->foreground = FG_BLUE;
				break;
			case BG_MAGENTA:
				vh->foreground = FG_MAGENTA;
				break;
			case BG_CYAN:
				vh->foreground = FG_CYAN;
				break;
			case BG_GREY:
				vh->foreground = FG_GREY;
		}
	} else vh->foreground = realfgcolor + highintensity + blink;
}

/* These functions return 0 if successful and 1 if error */
static inline char do_move_up(struct vcs_handle *vh,int count)
{
	if (count) {
		if ((vh->ypos - count) <= 1) {
			if (!vcs_setpos(vh,vh->xpos,1))
				return 1;
		} else {
			if (!vcs_setpos(vh,vh->xpos,vh->ypos-count))
				return 1;
		}
	}
	return 0;
}
static inline char do_move_down(struct vcs_handle *vh,int count)
{
	if (count) {
		if ((vh->ypos + count) >= vh->ysize) {
			if (!vcs_setpos(vh,vh->xpos,vh->ysize))
				return 1;
		} else {
			if (!vcs_setpos(vh,vh->xpos,vh->ypos+count))
				return 1;
		}
	}
	return 0;
}
static inline char do_move_left(struct vcs_handle *vh,int count)
{
	if (count) {
		if ((vh->xpos - count) <= 1) {
			if (!vcs_setpos(vh,1,vh->ypos))
				return 1;
		} else {
			if (!vcs_setpos(vh,vh->xpos-count,vh->ypos))
				return 1;
		}
	}
	return 0;
}
static inline char do_move_right(struct vcs_handle *vh,int count)
{
	if (count) {
		if ((vh->xpos + count) >= vh->xsize) {
			if (!vcs_setpos(vh,vh->xsize,vh->ypos))
				return 1;
		} else {
			if (!vcs_setpos(vh,vh->xpos+count,vh->ypos))
				return 1;
		}
	}
	return 0;
}
static inline char do_moveto(struct vcs_handle *vh,unsigned char *codestring)
{
	unsigned char *xs,*ys;
	int x,y;
	
	if (strchr(codestring,';')) {
		ys = strtok(codestring,";");
		xs = strtok(NULL,";");
		if (strlen(xs))
			x = atoi(xs);
		else
			x = vh->xpos;
		if (strlen(ys))
			y = atoi(ys);
		else
			y = vh->ypos;
		if (((x >= 1)&&(x <= vh->xsize))&&((y >= 1)&&(y <= vh->ysize))) {
			if (!vcs_setpos(vh,x,y))
				return 1;
		}
	}
	return 0;
}

/* This routine parses the attribute escape code */
static inline void do_set_attributes(struct vcs_handle *vh,unsigned char *codestring)
{
	unsigned char *tmp;
	
	if (strchr(codestring,';')) {
		if ((tmp = strtok(codestring,";"))) {
			if (strlen(tmp))
				do_colorchange(vh,atoi(tmp));
			while ((tmp = strtok(NULL,";"))) {
				if (strlen(tmp))
					do_colorchange(vh,atoi(tmp));
			}
		}
	} else {
		if (strlen(codestring))
			do_colorchange(vh,atoi(codestring));
	}
}

/* Writes a block of ansi graphics to the console at the current position.
 * If ansiblock is NULL, the ansi driver is initialized.  It should be
 * called with ansiblock=NULL before any i/o is performed.  The ansiblock
 * may contain broken codes at the end as long as they are resumed in the
 * next block.  The NULL call should also be done when i/o is finished.
 * Returns 0 if failed and nonzero if successful. */
extern int vcs_ansiprint(struct vcs_handle *vh,unsigned char *ansiblock,int delay)
{
	static int cbptr,ncbptr,savedx,savedy,nolinewrap;
	static unsigned char codebuf[64],noncodebuf[8194];
	register unsigned char *tmp = ansiblock;
	int delayloop;

	if (ansiblock == NULL) {
		if (ncbptr) {
			noncodebuf[ncbptr] = '\0';
			if ((nolinewrap)&&(strlen(noncodebuf) >= vh->xsize))
				noncodebuf[vh->xsize] = '\0';
			if (!vcs_printf(vh,noncodebuf))
				return 0;
			ncbptr = 0;
		}
		cbptr = 0;
		savedx = 0;
		savedy = 0;
		nolinewrap = 0;
		do_colorchange(vh,-255);
		return 1;
	}
	
	while (*tmp) {
		if ((*tmp == '\033')&&(*(tmp+1) == '[')) {
			codebuf[cbptr++] = *tmp;
			if (ncbptr) {
				noncodebuf[ncbptr] = '\0';
				if ((nolinewrap)&&(strlen(noncodebuf) >= vh->xsize))
					noncodebuf[vh->xsize] = '\0';
				if (!vcs_printf(vh,noncodebuf))
					return 0;
				ncbptr = 0;
			}
		} else {
			if (cbptr) {
				codebuf[cbptr++] = *tmp;
				if (strchr("JKABCDHsunmhl",*tmp)) {
					codebuf[cbptr-1] = '\0';
					switch (*tmp) {
						case 'J':
							if (!vcs_cls(vh))
								return 0;
							break;
						case 'K':
							if (!vcs_clreol(vh))
								return 0;
							break;
						case 'A':
							if (strlen(&codebuf[2])) {
								if (do_move_up(vh,atoi(&codebuf[2])))
									return 0;
							} else {
								if (do_move_up(vh,1))
									return 0;
							}
							break;
						case 'B':
							if (strlen(&codebuf[2])) {
								if (do_move_down(vh,atoi(&codebuf[2])))
									return 0;
							} else {
								if (do_move_down(vh,1))
									return 0;
							}
							break;
						case 'C':
							if (strlen(&codebuf[2])) {
								if (do_move_right(vh,atoi(&codebuf[2])))
									return 0;
							} else {
								if (do_move_right(vh,1))
									return 0;
							}
							break;
						case 'D':
							if (strlen(&codebuf[2])) {
								if (do_move_left(vh,atoi(&codebuf[2])))
									return 0;
							} else {
								if (do_move_left(vh,1))
									return 0;
							}
							break;
						case 'H':
							if (do_moveto(vh,&codebuf[2]))
								return 0;
							break;
						case 's':
							savedx = vh->xpos;
							savedy = vh->ypos;
							break;
						case 'u':
							if (savedx) {
								if (!vcs_setpos(vh,savedx,savedy))
									return 0;
							}
							break;
						case 'n':
						case 'h':
							break;
						case 'l':
							nolinewrap++;
							break;
						case 'm':
							do_set_attributes(vh,&codebuf[2]);
					}
					cbptr = 0;
				}
				if (cbptr >= 64)
					cbptr = 0;
			} else {
				if (ncbptr >= 8193) {
					noncodebuf[ncbptr] = '\0';
					if ((nolinewrap)&&(strlen(noncodebuf) >= vh->xsize))
						noncodebuf[vh->xsize] = '\0';
					if (!vcs_printf(vh,noncodebuf))
						return 0;
					ncbptr = 0;
				} else noncodebuf[ncbptr++] = *tmp;
			}
		}
		tmp++;
		if (delay)
			for (delayloop=0;delayloop<=delay;delayloop++) {}
	}

	return 1;
}
