/***************************************************************************
    This file is part of the CheeseTronic Music Tools
    url                  : http://reduz.com.ar/cheesetronic
    copyright            : (C) 2003 by Juan Linietsky
    email                : coding@reduz.com.ar
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "loader_wav.h"

#include <iostream>

struct format_chunk {

   short int channels;
   int sample_rate;
   short int bytes_per_sample;
};

struct data_chunk {

   int length;
   unsigned char* samples;
};

struct smpl_chunk {

   int rate;
};

struct multi_chunk {

   multi_chunk() : has_riff(0), has_format(0), has_data(0), has_smpl(0) {};

   bool has_riff, has_format, has_data, has_smpl;

   format_chunk format;
   data_chunk data;
   smpl_chunk smpl; // not implemented yet***
};

// a little mac to make the code a bit cleaner
static int load_chunk( char chunk_id[4],File_Reader& file_read, multi_chunk& Dest ) {


   // check for the types of chunks we know
   if( !memcmp( chunk_id, "RIFF", 4 ) ) {

	//printf("riff at %i\n",file_read.get_file_pos());

      if( Dest.has_riff ) {

         ERROR("load chunk ERRORing: file has many riff headers, that's odd but I don't care I'm just ignoring the extras");
         return 0;
      };

      int length=file_read.get_dword();

      char WAVE[4];

      file_read.get_byte_array((Uint8*)WAVE,4);

      if( memcmp( WAVE, "WAVE", 4 ) )
         ERROR("load chunk ERRORing: WAVE != WAVE, this sample might be from another universe, please zip it and send 10 times to reduz@anime.com.ar");


      Dest.has_riff = 1;
   }
   else if( !memcmp( chunk_id, "fmt ", 4 ) ) {

	//printf("format at %i\n",file_read.get_file_pos());

      if( Dest.has_format ) {

         ERROR("load chunk ERRORing: file has already a format chunk, ignoring");
         return 0;
      };

      int length;
      length=file_read.get_dword();

      if( length != 0x10 ) ERROR("load chunk fatal: format chunk length ain't right");

      short int one;
      one=file_read.get_word();

      if( one != 1 ) ERROR("load chunk fatal: one isn't one, what is WRONG with microsoft!");

      Dest.format.channels=file_read.get_word();

      if( Dest.format.channels < 1 || Dest.format.channels > 2 )
         ERROR("load chunk fatal: this file has " << Dest.format.channels << " channels, I can't deal with it sorry");

      Dest.format.sample_rate=file_read.get_dword();
      if( Dest.format.sample_rate < 0 )
         ERROR("load chunk fatal: this file has got some wack sample rate going on");

      int bytes_per_second; // ignored
      bytes_per_second=file_read.get_dword();
      Dest.format.bytes_per_sample=file_read.get_word();

      if( Dest.format.channels == 2 )
	      Dest.format.bytes_per_sample/=2;

	if( Dest.format.bytes_per_sample != 1 && Dest.format.bytes_per_sample != 2 )
		ERROR("load chunk fatal: I can only load 8bit or 16bit samples, this mono sample says its " << Dest.format.bytes_per_sample * 8 << "bit");

      short int bits_per_sample; // ignored
      bits_per_sample=file_read.get_word();

      Dest.has_format = 1;
   }
   else if( !memcmp( chunk_id, "data", 4 ) ) {

	//printf("data at %i\n",file_read.get_file_pos());

      if( Dest.has_data ) {

         ERROR("load chunk ERRORing: file has more than one data chunks, ignoring secondary ones.");
         return 0;
      };

      if( !Dest.has_format )
         ERROR("load chunk fatal: format chunk not defined before data chunk, I don't know how to load this sorry!");

      Dest.has_data = 1;
      Dest.data.length=file_read.get_dword();

      if( Dest.data.length < 0 )
         ERROR("load chunk fatal: size < 0 loading sample data, that ain't right");

      Dest.data.samples = (Uint8*) malloc( Dest.data.length );

      if (Dest.format.bytes_per_sample==1)
      	file_read.get_byte_array((Uint8*)Dest.data.samples,Dest.data.length);
      else
      	file_read.get_word_array((Uint16*)Dest.data.samples,Dest.data.length/2);

      Dest.has_data = 1;
   }
/*   else if( !memcmp( chunk_id, "smpl", 4 ) ) {

      if( Dest.has_smpl ) {

         ERROR("load chunk ERRORing: file has more than one smpl chunk");
         return 0;
      };


   }
   else {*/

   else {

      // unknonw chunk format, skip over it
      int skip;

        skip=file_read.get_dword();
      	//printf("unknown block %c%c%c%c\n",chunk_id[0],chunk_id[1],chunk_id[2],chunk_id[3]);
        file_read.seek( file_read.get_file_pos() + skip );
//      return 0;
//      ERROR("Loader_WAV.cpp: load sample: load chunk fatal: unrecognized chunk type " << chunk_id);
   };

   return 0;
};


Sample_Data* Loader_WAV::load_sample(const char *p_filename) {

	if (file_read.open(p_filename))
		return NULL;



	// try and open the file

	// our detection scheme is simple, we just need to see a riff chunk first thing.
	bool first_chunk_detection_flag = 1;

	// gobble the file up while we still can; putting the chunks into our multichunk collection
	multi_chunk Header;
	while(1) {

	char chunk_id[4];
	file_read.get_byte_array((Uint8*)chunk_id, 4);
	// if this is the first chunk, we'll "detect" the file format by checking if
	// the first chunk id is a RIFF header.
	if( first_chunk_detection_flag ) {

		if( memcmp( chunk_id, "RIFF", 4 ) ) {

		// no worries, it just means we don't know what the heck this is.
		file_read.close();
		return NULL;
		};

		first_chunk_detection_flag = 0;
	};

	if( load_chunk( chunk_id, file_read,Header ) ) {

		// oop!
		file_read.close();
		if( Header.has_data ) free(Header.data.samples); // don't leak memory
		ERROR("load sample fatal: I thought I could load this sample, now I realize that I can't. I'm sorry.");
		return NULL; // probably.. ok well
	};

		if( Header.has_riff && Header.has_format && Header.has_data )
			break;
		//if (file_read.eof_reached())
		//	break;

	};

	// check that we aren't missing any of our chunks
	if( !Header.has_riff || !Header.has_format || !Header.has_data ) {

		ERROR("load sample fatal: this WAV file seems to be incomplete, I can't load it");
		file_read.close();
		return NULL;
	};

	file_read.close();

	Sample_Data *SD= new Sample_Data;

	SD->set_c5_freq(Header.format.sample_rate);

	// set up the data; we'll just do this two ways depending on stereo or mono
	if( Header.format.channels == 1 ) {

		// mono; the easy case
		int size=(Header.data.length / Header.format.bytes_per_sample);
		bool is16b=(Header.format.bytes_per_sample == 2);
		SD->set_data_ptr ( (Sint16*) Header.data.samples, size, is16b);
	} else {

		// stereo is a bit more complicated;
		int size=(Header.data.length / Header.format.bytes_per_sample)/2;
		bool is16b=(Header.format.bytes_per_sample == 2);
		//printf("size is %i, length %i, bps %i\n",size,Header.data.length,Header.format.bytes_per_sample);

		// we'll introduce a bizarre but useful kind of behaviour, and decide what channel
		// to load based on the parity of the input index; we'll document this by saying
		// "if you want to load stereo samples, just load a stereo sample twice consecutively!
		// MAGIC!" baaah
		int channel = 0;
		void * dst_buffer;
		if( is16b ) {

			short int* Source = (short int*) Header.data.samples;

			Uint16 * Dest = (Uint16*)malloc(size*2);
			dst_buffer=Dest;
			for( int a = 0; a < size; ++a ) {

				short int X = Source[a * 2 + channel];
				*Dest++ = X;
			};


		} else {

			char* Source = (char*) Header.data.samples;

			char* Dest = (char*) malloc(size);
			dst_buffer=Dest;

			for( int a = 0; a < size; ++a ) {

				char X = Source[a * 2 + channel];
				*Dest++ = X;
			};

		};
		SD->set_data_ptr(dst_buffer,size,is16b);
		// we just copied all the data (had to to dis-intereleave the stereo sample)
		// so we'll now deallocate our other buffer;
		free(Header.data.samples);
	};

	if (!SD->is_16bit())
		SD->change_sign();


	return SD;
};


Loader::Error Loader_WAV::load_sample(const char *p_filename, int p_dest_index) {


	if (file_read.open(p_filename))
		return FILE_ERROR;



	// try and open the file

	// our detection scheme is simple, we just need to see a riff chunk first thing.
	bool first_chunk_detection_flag = 1;

	// gobble the file up while we still can; putting the chunks into our multichunk collection
	multi_chunk Header;
	while(1) {

	char chunk_id[4];
	file_read.get_byte_array((Uint8*)chunk_id, 4);
	// if this is the first chunk, we'll "detect" the file format by checking if
	// the first chunk id is a RIFF header.
	if( first_chunk_detection_flag ) {

		if( memcmp( chunk_id, "RIFF", 4 ) ) {

		// no worries, it just means we don't know what the heck this is.
		file_read.close();
		return FILE_FORMAT_NOT_RECOGNIZED;
		};

		first_chunk_detection_flag = 0;
	};

	if( load_chunk( chunk_id, file_read,Header ) ) {

		// oop!
		file_read.close();
		if( Header.has_data ) free(Header.data.samples); // don't leak memory
		ERROR("load sample fatal: I thought I could load this sample, now I realize that I can't. I'm sorry.");
		return HEADER_CORRUPT; // probably.. ok well
	};

		if( Header.has_riff && Header.has_format && Header.has_data )
			break;
		//if (file_read.eof_reached())
		//	break;

	};

	// check that we aren't missing any of our chunks
	if( !Header.has_riff || !Header.has_format || !Header.has_data ) {

		ERROR("load sample fatal: this WAV file seems to be incomplete, I can't load it");
		file_read.close();
		return HEADER_CORRUPT;
	};

	file_read.close();

	// we've got everything, now what remains is to pass it on.
	Sample* S = song->get_sample(p_dest_index);

	// setup the header
	S->reset();
	S->name = p_filename;
	S->filename = p_filename;
	S->in_use = true;

	Sample_Data *SD=&S->data;

	SD->set_c5_freq(Header.format.sample_rate);

	// set up the data; we'll just do this two ways depending on stereo or mono
	if( Header.format.channels == 1 ) {

		// mono; the easy case
		int size=(Header.data.length / Header.format.bytes_per_sample);
		bool is16b=(Header.format.bytes_per_sample == 2);
		SD->set_data_ptr ( (Sint16*) Header.data.samples, size, is16b);
	} else {

		// stereo is a bit more complicated;
		int size=(Header.data.length / Header.format.bytes_per_sample)/2;
		bool is16b=(Header.format.bytes_per_sample == 2);
		//printf("size is %i, length %i, bps %i\n",size,Header.data.length,Header.format.bytes_per_sample);

		// we'll introduce a bizarre but useful kind of behaviour, and decide what channel
		// to load based on the parity of the input index; we'll document this by saying
		// "if you want to load stereo samples, just load a stereo sample twice consecutively!
		// MAGIC!" baaah
		int channel = 0;
		void * dst_buffer;
		if( is16b ) {

			short int* Source = (short int*) Header.data.samples;

			Uint16 * Dest = (Uint16*)malloc(size*2);
			dst_buffer=Dest;
			for( int a = 0; a < size; ++a ) {

				short int X = Source[a * 2 + channel];
				*Dest++ = X;
			};


		} else {

			char* Source = (char*) Header.data.samples;

			char* Dest = (char*) malloc(size);
			dst_buffer=Dest;

			for( int a = 0; a < size; ++a ) {

				char X = Source[a * 2 + channel];
				*Dest++ = X;
			};

		};
		SD->set_data_ptr(dst_buffer,size,is16b);
		// we just copied all the data (had to to dis-intereleave the stereo sample)
		// so we'll now deallocate our other buffer;
		free(Header.data.samples);
	};

	if (!SD->is_16bit())
		SD->change_sign();


	return SUCCESS;
};

bool Loader_WAV::test(const char *p_filename) {

   return false;
};

// the following methods shouldn't be called since this is just a sample loader
// (test returns false letting the tracker know that we don't load songs)

Loader::Error Loader_WAV::load(const char *p_filename,bool p_load_patterns) {

   ERROR("load: unexpected method invocation");
   return FILE_FORMAT_NOT_RECOGNIZED; // not valid to load a song file as a sample
};

int Loader_WAV::get_amount_of_samples() {

   ERROR("get_amount_of_samples: unexpected method invocation");
   return 0;
};

Sample_Data *Loader_WAV::get_sample_data(int p_sample_index) {

   ERROR("get_sample_data: unexpected method invocation");
   return NULL;
};

string Loader_WAV::get_sample_name(int p_sample_index) {

   ERROR("get_sample_name: unexpected method invocation");
   return "";
};

void Loader_WAV::add_sample_to_song(int p_sample_index,int p_dest_index,bool create_instrument) {

   ERROR("add_sample_to_song: unexpected method invocation");
};

Loader::Error Loader_WAV::load_samples_from_instrument(const char *p_filename) {

   ERROR("load_samples_from_instrument: unexpected method invocation");
   return FILE_ERROR;
};

Loader::Error Loader_WAV::load_instrument(const char *p_filename,int p_dest_index) {

   ERROR("load_instrument: unexpected method invocation");
   return FILE_ERROR;
};

void Loader_WAV::transfer_data_to_song() {

   ERROR("transfer_data_to_song: unexpected method invocation");
};

void Loader_WAV::free_info(bool free_sampledata) {

   ERROR("free_info: unexpected method invocation");
};

Loader_WAV::Loader_WAV() {

	format_name="Microsoft WAV";

}
Loader_WAV::~Loader_WAV() {

}

