/*************************************************************************************************
*
*	Title:	Tunnel.cpp
*	Desc:	Simple 2D Tunnel effect
*	
*	Note:	1) for more information check the demo sources. this is beyond the scope of these
*				tutes.
*			
**************************************************************************************************/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <memory.h>
#include "VVideo.h"

/*************************************************************************************************/
// External Data
/*************************************************************************************************/

/*************************************************************************************************/
// Global Data
/*************************************************************************************************/

/*************************************************************************************************/
// Module Data
/*************************************************************************************************/

static Color32_t	*m_TextureMap;
static int			*m_TunnelLUT;

/*************************************************************************************************/
// Functions
/*************************************************************************************************/

/*************************************************************************************************
*
*	Function:	Tunnel_Init()
*
*	Desc:		Inits tunnel datat
*
*	Notes:		1) Computes the lookup table
*				2) This is a generic tunnel, you can just as easily make a textured torus or
*					generaly any solvable Function of (x,y,z)
*
***************************************************************************************************/
void Tunnel_Init( void )
{
	float x, y;
	float Dx, Dy, Dz;
	float Dist;
	float t;
	float Ix, Iy, Iz;
	float tu, tv;
	float Radius, Radius2;
	int *TunnelLUT;
	int i;
	FILE *File;

	// allocate texture map
	m_TextureMap = (Color32_t *)malloc( sizeof(Color32_t) * 256 * 256 );

	// load it
	File = fopen( "Texture.raw", "rb" );
	if (File)
	{
		for (i=0; i < 256*256; i++)
		{
			m_TextureMap[i].r = getc( File );
			m_TextureMap[i].g = getc( File );
			m_TextureMap[i].b = getc( File );
		}
		
		fclose( File );
	}

	// set cylender radius
	Radius = 1.0f;
	Radius2 = Radius * Radius;
	

	// allocate LUT
	m_TunnelLUT = (int *)malloc( sizeof(int) * 320 *240 );

	TunnelLUT = m_TunnelLUT;

	// generate lookup
	for (y = -1.0f; y < 1.0f; y += (2.0f/239.0f) )
	{
		for (x = -1.0f; x < 1.0f; x += (2.0f/319.0f) )
		{
			// calculate ray direction
			Dx = x;
			Dy = y;
			Dz = 1.0f;

			// normalize
			Dist = 1.0f / (float)sqrt( Dx*Dx + Dy*Dy + Dz*Dz );
			Dx *= Dist;
			Dy *= Dist;
			Dz *= Dist;

			// calculate intersection with object
			//
			// Eq Cylinder x^2 + y^2 = 1									Eq (1)
			//
			// Paremetric Eq of our ray -> (0,0,0) + (Dx, Dy, Dz)*t = 0		Eq (2)
			//
			// Substue (2) into Eq (1)
			//
			// -> ( Dx*t )^2 + (Dy*t)^2	= Radius^2
			//
			// -> t^2 ( Dx^2 + Dy^2 )	= Radius^2
			//
			// ->					  t = sqrt(Radius^2 / (Dx^2 - Dy^2) ) 
			//
			t = (float)sqrt( Radius2  / (Dx*Dx + Dy*Dy) );

			// get intersection point
			Ix = Dx * t;
			Iy = Dy * t;
			Iz = Dz * t;

			// use a sort of cylndrical texture projection

			// use how far down the z axis the point is as the y texture compenent
			tv = (float)fabs(Iz)* 0.2f;
			i=0;
			while ( tv > 1.0f)
			{
				i++;
				tv -= 1.0f;
			}

			// use the angle around the cylenders z axis as the x texture component
			tu =(3.1415f + (float)atan2( Iy, Ix ))/ (3.1415f*2.0f);
			if (tu < 0.0f) tu = 0.0f;
			if (tv > 1.0f) tv = 1.0f;

			// stop tunnel at a certain distance
			if (i <= 2)
			{
				// save address
				*TunnelLUT = (int)(tu*256.0f) + 256*(int)(tv * 256);
			}
			else
			{
				*TunnelLUT = 0;
			}

			// next tunnel LUT entry
			TunnelLUT++;
		}
	}
}

/*************************************************************************************************
*
*	Function:	Tunnel_Kill()
*
*	Desc:		Frees any resources
*
*	Notes:		
*
***************************************************************************************************/
void Tunnel_Kill( void )
{
	// release texture map memory
	if (m_TextureMap)
	{
		free( m_TextureMap );
		m_TextureMap = NULL;
	}

	// release LUT memory
	if (m_TunnelLUT)
	{
		free( m_TunnelLUT );
		m_TunnelLUT = NULL;
	}
}


/*************************************************************************************************
*
*	Function:	Tunnel_Update()
*
*	Desc:		Updates one frame of the tunnel effect
*
*	Notes:		
*
***************************************************************************************************/
void Tunnel_Update( void )
{
	Color32_t *VPage;
	int i;
	int *TunnelLUT;

	// get display page
	VPage = VV_GetAddress();

	// set up pointers
	TunnelLUT = m_TunnelLUT ;

	// loop over every pixel
	i = 320*240;
	do
	{
		// end of the tunnel?
		if (*TunnelLUT)
		{
			// copy pixel
			VPage->r = (m_TextureMap + *TunnelLUT)->r;
			VPage->g = (m_TextureMap + *TunnelLUT)->g;
			VPage->b = (m_TextureMap + *TunnelLUT)->b;
		}
		else
		{
			// back ground color
			VPage->r = 0;
			VPage->g = 0;
			VPage->b = 0;
		}

		// next
		VPage++;
		TunnelLUT++;

		// decrepent counter
		i--;

	} while(i > 0);

	// copy first row to the bottom
	memcpy( m_TextureMap + 255*256, m_TextureMap, sizeof(Color32_t)*256 );

	// move the texture upward 1 row
	for (i=0; i < 255; i++)
	{
		memcpy( m_TextureMap + i*256, m_TextureMap + (i+1)*256, sizeof(Color32_t)*256 );
	}
}