/*************************************************************************************************
*
*	Title:	GDI.cpp
*	Desc:	Provides a GDI based virtual frame buffer interface
*	
*	Note:	1) Will use a DIB of the same bit depth as the desktop
*			
**************************************************************************************************/

#include <Windows.h>
#include <memory.h>
#include "GDI.h"
#include "Winmain.h"
#include "VVideo.h"

/*************************************************************************************************/
// External Data
/*************************************************************************************************/

/*************************************************************************************************/
// Global Data
/*************************************************************************************************/

/*************************************************************************************************/
// Module Data
/*************************************************************************************************/

static	HWND				m_hWnd;

// our virtual page
static	Color32_t			*m_VPage = NULL;
static	unsigned short		*m_WPage = NULL;
static unsigned int			*m_DPage = NULL;

static int					m_Bpp;

static	char 				m_bibuf[ sizeof(BITMAPINFOHEADER) + 3*4 ];
static	BITMAPINFO      	&m_bi = *(BITMAPINFO*)&m_bibuf;

/*************************************************************************************************/
// Functions
/*************************************************************************************************/

/*************************************************************************************************
*
*	Function:	GDI_Open()
*
*	Desc:		Sets up our DIB and Virtaul page
*
*	Notes:		
*
***************************************************************************************************/
int GDI_Open( void )
{

	HDC hDC;

	// get the windows Device Contex
	hDC = GetDC( g_hWnd );
	if (hDC == NULL)
	{
		return 1;
	}

	// work out what the desktop bpp is
	m_Bpp  =  GetDeviceCaps( hDC, BITSPIXEL);

	// close resource
	ReleaseDC( g_hWnd, hDC );

	// Set most of the header info fields
	memset( &m_bi.bmiHeader, 0, sizeof(m_bi.bmiHeader) );
	m_bi.bmiHeader.biSize = sizeof(m_bi.bmiHeader);
	m_bi.bmiHeader.biWidth = 320;
	m_bi.bmiHeader.biHeight = -240;			// NOTE: -240. this is because DIB's are upside down
	m_bi.bmiHeader.biPlanes = 1;
	m_bi.bmiHeader.biBitCount = m_Bpp;
	m_bi.bmiHeader.biCompression = BI_BITFIELDS;

	// create a dib the same bit depth as the desktop, as GDI color space conversion are very slow
	switch (m_Bpp)
	{
		// 16bpp
		case 16:
			// set format 555
			((unsigned long*)m_bi.bmiColors)[0]=0x00007c00;
			((unsigned long*)m_bi.bmiColors)[1]=0x000003e0;
			((unsigned long*)m_bi.bmiColors)[2]=0x0000001f;

			// allocate 16 GDI memory
			m_WPage = (unsigned short *)malloc( 320 * 240 * sizeof(unsigned short) );

			break;
	
		// 24/32bpp
		case 24: 
		case 32:
			// 32bit color 888
			((unsigned long*)m_bi.bmiColors)[0]=0x00ff0000;
			((unsigned long*)m_bi.bmiColors)[1]=0x0000ff00;
			((unsigned long*)m_bi.bmiColors)[2]=0x000000ff;	

			// allocate 32bit GDI memory
			m_DPage = (unsigned int *)malloc( 320 * 240 * sizeof(unsigned int) );
			break;

		// unnknown 
		default:
			return 1;
	}
		
	// allocate vpage
	m_VPage = (Color32_t *)malloc( 320 * 240 * sizeof(Color32_t) );
	if (m_VPage == NULL)
	{
		return 1;
	}

	return 0;
}

/*************************************************************************************************
*
*	Function:	GDI_Close()
*
*	Desc:		Releases the Virtual pages memory
*
*	Notes:		
*
***************************************************************************************************/
void GDI_Close( void )
{
	// check we allocated it first
	if (m_VPage)
	{
		free( m_VPage );
	}
}

/*************************************************************************************************
*
*	Function:	GDI_Flip()
*
*	Desc:		Copyies the virtual page to the window
*
*	Notes:		
*
***************************************************************************************************/
void GDI_Flip( void )
{
	RECT client;
	HDC hDC;
	int i;
	unsigned short *WData;
	unsigned int *DData;
	Color32_t *VPData;
	unsigned int *DIBData;

	// convert the page to whatever format the desktop is at
	switch (m_Bpp)
	{
		// 16bpp
		case 16:

			// set pointers to top of destination and source memory
			WData = m_WPage;
			VPData = m_VPage;

			// for all pixels convert
			for (i=0; i < 320*240; i++)
			{
				// convert VPage 32bpp 8888 data into 16bit 555
				*WData = ((VPData->b>>3)&0x001f) + ((VPData->g<<2)&0x03e0) + ((VPData->r<<7)&0x7c00);

				// next pixel
				WData++;
				VPData++;
			}

			// set the DIB's source data
			DIBData = (unsigned int *)m_WPage;
			break;

		// 24/32 bpp
		case 24:
		case 32:
			// set up pointers to destination and source
			DData = m_DPage;
			VPData = m_VPage;

			// for all pixels
			for (i=0; i < 320*240; i++)
			{
				// convert from 32bpp ARGB to BGRA
				*DData = ((VPData->b<<24)&0xff000000) + ((VPData->g<<16)&0x00ff0000) +((VPData->r<<8)&0x0000ff00);

				// next pixel
				DData++;
				VPData++;
			}

			// set DIB's Source data
			DIBData = m_DPage;
			break;
	}

	//
	// draw it to our window
	//

	// get the windows Device Contex
	hDC = GetDC( g_hWnd );

	// if its avaliable
	if (hDC != NULL)
	{

		// Get the the window dimentions
		GetClientRect(g_hWnd, &client);

		// Streach it
		StretchDIBits(	hDC, 
						0, 				// Destination top left hand corner X Position
						0, 				// Destination top left hand corner Y Position
						client.right, 	// Destinations Width
						client.bottom, 	// Desitnations height

						0, 				// Source top left hand corner's X Position
						0, 				// Source top left hand corner's Y Position
						320, 			// Sources width
						240,			// Sources height

						DIBData, 		// Source's data
						&m_bi, 			// Bitmap Info
						DIB_RGB_COLORS, // operations
						SRCCOPY);		

		// now relese the Device Contex.
		ReleaseDC( g_hWnd, hDC );
	}
}

/*************************************************************************************************
*
*	Function:	GDI_GetAddress()
*
*	Desc:		Gets the virtual page's top left hand corners memory address
*
*	Notes:		
*
***************************************************************************************************/
Color32_t *GDI_GetAddress( void )
{
	return m_VPage;
}
